﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public class MapProviderServiceToCrm : MapperRelatedBase
    {
        private readonly MapCareSiteToCrm _careSiteMapper;

        public MapProviderServiceToCrm(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper, MapCareSiteToCrm careSiteMapper)
            : base(ppmsContextHelper, ppmsHelper)
        {
            _careSiteMapper = careSiteMapper;
        }

        public async Task<ppms_providerservice> MapInsert(ProviderService service, Account provider)
        {
            // Map entity
            var entity = new ppms_providerservice { Id = Guid.NewGuid() };

            // Specialty
            if (!string.IsNullOrEmpty(service.CodedSpecialty))
            {
                var specialty = GetTaxonomy(service.CodedSpecialty);
                if (specialty != null) entity.ppms_specialty = specialty;
            }

            // Care site
            if (service.Location != null) await SetCareSite(service.Location, entity);
            
            // Station Number
            //if (!string.IsNullOrEmpty(service.StationNumber))
            //{
            //    var careSite = await GetCareSiteByStationNumber(service.StationNumber);
                
            //    if (careSite != null) entity.ppms_caresite = entity.ToEntityReference();
            //}

            // Organization
            if (!string.IsNullOrEmpty(service.OrganizationNpi))
            {
                var org = await GetProviderByNpi(service.OrganizationNpi);
                if (org != null) entity.ppms_Organization = org.ToEntityReference();
            }

            // shorthand value for network lookup
            string shorthand = string.Empty;

            if (!ForVaNetwork && service.RelationshipSpecified)
            {
                var relationshipValue = service.Relationship.ToString();
                shorthand = $"{NetworkShorthand}_{relationshipValue}";

                // Relationship
                var relationship = GetVaProviderRelationship(relationshipValue);
                if (relationship != null)
                {
                    entity.ppms_VAProviderRelationshipId = relationship;
                }
            }
            else
            {
                shorthand = $"{NetworkShorthand}";
            }

            // Affiliation
            entity.ppms_network = GetNetwork(shorthand);

            // Set station number
            if (!string.IsNullOrEmpty(service.StationNumber))
            {
                var facility = await PpmsHelper.GetFacilityByStationNumber(service.StationNumber);
                if (facility != null) entity.ppms_Facility = facility;
            }

            return entity;
        }

        public async Task<ppms_providerservice> MapUpdate(ProviderService entity, Account provider)
        {
            ppms_providerservice crmEntity;
            // get entity
            if (string.IsNullOrEmpty(entity.CorrelationId))
            {
                // Get related entities with matching name
                crmEntity = GetCrmEntity(provider, entity);
            }
            else
            {
                // get entity, if CorrelationId provided
                crmEntity = GetCrmEntity(provider, entity.CorrelationId);
            }

            // matching account not found, insert new record
            if (crmEntity == null)
            {
                return await MapInsert(entity, provider);
            }

            var newEntity = new ppms_providerservice
            {
                Id = crmEntity.Id,
                ppms_name = crmEntity.ppms_name
            };

            // Affiliation
            if (entity.RelationshipSpecified)
                crmEntity.ppms_network = GetNetwork(NetworkShorthand, entity.Relationship.ToString());
            else
                crmEntity.ppms_network = GetNetwork(NetworkShorthand);

            // Map fields
            if (!string.IsNullOrEmpty(entity.CodedSpecialty))
            {
                var specialty = GetTaxonomy(entity.CodedSpecialty);
                if (specialty != null) newEntity.ppms_specialty = specialty;
            }

            if (entity.Location != null) await SetCareSite(entity.Location, newEntity);
            else if (!string.IsNullOrEmpty(entity.StationNumber)) await SetCareSite(crmEntity, entity.StationNumber);

            if (entity.RelationshipSpecified)
            {
                // Relationship
                var relationship = GetVaProviderRelationship(entity.Relationship.ToString());
                if (relationship != null)
                {
                    crmEntity.ppms_VAProviderRelationshipId = relationship;
                }
            }

            // Set station number
            if (!string.IsNullOrEmpty(entity.StationNumber))
            {
                var facility = await PpmsHelper.GetFacilityByStationNumber(entity.StationNumber);
                if (facility != null) crmEntity.ppms_Facility = facility;
            }

            return newEntity;
        }

        private async Task<ppms_caresite> SetCareSite(CareSite site, ppms_providerservice targetService)
        {
            // determine if care site exists
            ppms_caresite entity;
            if (string.IsNullOrEmpty(site.CorrelationId)) entity = await CheckForExistingCareSite(site);
            else
            {
                using (var context = await PpmsContextHelper.GetContextAsync())
                {
                    entity = context.ppms_caresiteSet.FirstOrDefault(a => a.Id == new Guid(site.CorrelationId));
                }
            }

            if (entity == null)
            {
                // Set reference values for child mapper
                _careSiteMapper.References = References;

                // Care site not found, map as insert
                entity = await _careSiteMapper.MapInsert(site);

                if (entity != null)
                {
                    // Set relationship to service
                    targetService.ppms_caresite_providerservice_caresite = entity;
                    
                    // Add to added entity list for future comparisons
                    AddModifiedEntity(entity);
                }
            }
            else targetService.ppms_caresite = entity.ToEntityReference();

            return entity;
        }

        private async Task<ppms_caresite> SetCareSite(ppms_providerservice entity, string stationNumber)
        {
            ppms_caresite careSite = null;

            // Station Number
            if (!string.IsNullOrEmpty(stationNumber))
            {
                careSite = await GetCareSiteByStationNumber(stationNumber);
                if (careSite != null)
                {
                    entity.ppms_caresite = entity.ToEntityReference();
                }
            }

            return careSite;
        }

        private ppms_providerservice GetCrmEntity(Account provider, string id)
        {
            var list = provider.ppms_account_ppms_providerservice;
            if (list == null) return null;

            var ppmsProviderservices = list as ppms_providerservice[] ?? list.ToArray();

            return ppmsProviderservices.Any() ? ppmsProviderservices.FirstOrDefault(x => x.Id.ToString() == id) : null;
        }

        private ppms_providerservice GetCrmEntity(Account provider, ProviderService providerService)
        {
            IEnumerable<ppms_providerservice> list = null;

            // Check for existing related entities
            if (ParentalReference != null)
            {
                var parent = (Account)ParentalReference;
                if (parent != null)
                    list = parent.ppms_account_ppms_providerservice;
            }

            if (list == null) return null;

            var ppmsProviderServices = list as ppms_providerservice[] ?? list.ToArray();

            var address = providerService.Location.SiteAddress;

            //return ppmsProviderServices.Any() ? 
            //    ppmsProviderServices.FirstOrDefault(x => x.ppms_ProviderRelationship.Name == providerService.Relationship.ToString()
            //        && x.ppms_caresiteaddress != null
            //        && x.ppms_caresiteaddress.Contains(address.Address1)
            //        && x.ppms_caresitecity == address.City
            //        && x.ppms_caresitestateprovince == address.State
            //        && x.ppms_caresitezipcode == address.PostalCode
            //    ) 
            //    : null;

            foreach (var item in ppmsProviderServices)
            {
                if ((item.ppms_VAProviderRelationshipId != null && providerService.RelationshipSpecified
                        && item.ppms_VAProviderRelationshipId.Name == providerService.Relationship.ToString())
                    && (item.ppms_caresiteaddress != null
                        && item.ppms_caresiteaddress.Contains(address.Address1)
                        && item.ppms_caresitecity == address.City
                        && item.ppms_caresitestateprovince == address.State
                        && item.ppms_caresitezipcode == address.PostalCode))
                {
                    return item;
                }
            }

            // matching item not found
            return null;
        }

        private async Task<ppms_caresite> CheckForExistingCareSite(CareSite careSite)
        {
            ppms_caresite match = null;
            var address = careSite.SiteAddress;

            if (address != null)
            {
                // Check for match in current batch of updates
                var activeList = ModifyEntityList;
                if (activeList != null && activeList.Any())
                {
                    var matchList = activeList.Where(x => x.GetAttributeValue<string>("ppms_name") == careSite.Name);
                    match = (ppms_caresite)matchList.FirstOrDefault(x => x.GetAttributeValue<string>("ppms_address_compositeid") == address.CompositeId);
                }

                // If no match found, search for existing entity
                if (match == null)
                {
                    match = await GetCareSiteByComposite(careSite);
                }
            }

            // No match found
            return match;
        }

        private async Task<ppms_caresite> GetCareSiteByStationNumber(string stationNumber)
        {
            using (var context = await PpmsContextHelper.GetContextAsync())
            {
                return context.ppms_caresiteSet.FirstOrDefault(s => s.ppms_stationnumber == stationNumber);
            }
        }

        private async Task<ppms_caresite> GetCareSiteByComposite(CareSite careSite)
        {
            if (careSite == null) return null;

            using (var context = await PpmsContextHelper.GetContextAsync())
            {
                var matchList = context.ppms_caresiteSet.Where(c => c.ppms_name == careSite.Name);
                if (careSite.SiteAddress != null)
                    return matchList.FirstOrDefault(a => a.ppms_address_compositeid.Contains(careSite.SiteAddress.CompositeId));
                else
                    return matchList.FirstOrDefault();
            }
        }

        private async Task<IEnumerable<ppms_caresite>> GetCareSitesByComposite(CareSite careSite)
        {
            if (careSite == null) return null;

            using (var context = await PpmsContextHelper.GetContextAsync())
            {
                var matchList = context.ppms_caresiteSet.Where(c => c.ppms_name == careSite.Name);
                if (careSite.SiteAddress != null)
                    return matchList.Where(a => a.ppms_address_compositeid.Contains(careSite.SiteAddress.CompositeId));
                else
                    return matchList;
            }
        }

        private EntityReference GetVaProviderRelationship(string relationshipType)
        {
            var result = References.VaProviderRelationships.FirstOrDefault(s => s.Value == relationshipType);
            if (result != null)
                return result.ToEntityReference("ppms_vaproviderrelationship");
            // Not found
            return null;
        }

        private static ProviderService ConvertEntity<T>(T entity)
        {
            return (ProviderService)Convert.ChangeType(entity, typeof(ProviderService));
        }

        public override async Task<Entity> MapUpdate<T>(T entity, Entity parent)
        {
            return await MapUpdate(ConvertEntity(entity), (Account)parent);
        }

        public override async Task<Entity> MapInsert<T>(T entity, Entity parent)
        {
            return await MapInsert(ConvertEntity(entity), (Account)parent);
        }

        public override async void AddChildrenToProvider(IList<Entity> entities, Entity parent)
        {
            if (entities == null) return;

            if (IsWithinContext)
            {
                bool isAddingRelationship;

                // Make sure provider entity is being tracked
                if (!Context.IsAttached(parent)) Context.Attach(parent);

                var newRelationship = new Relationship("ppms_account_ppms_providerservice");
                IEnumerable<ppms_providerservice> children = null;
                var account = (Account)ParentalReference;
                if (account != null)
                {
                    children = account.ppms_account_ppms_providerservice;
                }

                foreach (var item in entities)
                {
                    isAddingRelationship = true;

                    // If parental reference exists, make sure the child is not associated to parent
                    if (children != null && children.Any(a => a.Id == item.Id))
                    {
                        isAddingRelationship = false;
                        if (!Context.IsAttached(item))
                        {
                            Context.Attach(item);
                            Context.UpdateObject(item);
                        }
                    }

                    if (isAddingRelationship)
                    {
                        Context.AddRelatedObject(parent, new Relationship("ppms_account_ppms_providerservice"), item);

                        // Add related care site
                        var csRel = new Relationship("ppms_caresite_providerservice_caresite");
                        if (item.RelatedEntities != null && item.RelatedEntities.ContainsKey(csRel))
                        {
                            var careSites = item.RelatedEntities[csRel];
                            if (careSites != null && careSites.Entities != null && careSites.Entities.Count > 0)
                            {
                                foreach (var entity in careSites.Entities)
                                {
                                    Context.AddRelatedObject(item, new Relationship("ppms_caresite_providerservice_caresite"), entity);
                                }
                            }
                        }
                    }
                }
            }
            else
            {
                if (entities.Count <= 0) return;

                var account = (Account)parent;
                if (account != null) account.ppms_account_ppms_providerservice = ConvertEntityList<ppms_providerservice>(entities);
            }

            await Task.Run(() => { });
        }

        public override IEnumerable<SetStateRequest> MapDelete<T>(IList<T> entities, Entity parent)
        {
            if (entities == null || !entities.Any()) return null;

            // Check provider
            var provider = (Account)parent;
            if (provider == null) return null;

            var list = new List<ppms_providerservice>();
            IEnumerable<ppms_providerservice> matches = null;
            ProviderService entity = null;

            // Map schema entities for delete
            foreach (var item in entities)
            {
                entity = ConvertEntity(item);

                // Determine items to be deactivated
                if (string.IsNullOrEmpty(entity.CorrelationId))
                {
                    // No correlation id specified, match by properties
                    // Get selected relationship
                    EntityReference rel = null;
                    if (entity.RelationshipSpecified)
                        rel = GetRelationship(entity.Relationship.ToString());

                    // Get care site reference
                    var caresites = GetCareSitesByComposite(entity.Location).GetAwaiter().GetResult();
                    IEnumerable<EntityReference> csRef = caresites.ToList().ConvertAll<EntityReference>(c => new EntityReference("ppms_caresite", c.Id));

                    if (caresites != null)
                    {
                        if (rel == null)
                        {
                            // No relationship provided, match by specialty, no relationship, and care site
                            matches = provider.ppms_account_ppms_providerservice.Where(
                                p => p.StateCode == ppms_providerserviceState.Active
                                && p.ppms_specialtycode == entity.CodedSpecialty
                                && p.ppms_VAProviderRelationshipId == null
                                && (p.ppms_caresite != null && csRef.Contains(p.ppms_caresite))
                            );
                        }
                        else
                        {
                            // match by specialty, relationship, and care site
                            matches = provider.ppms_account_ppms_providerservice.Where(
                                p => p.StateCode == ppms_providerserviceState.Active
                                && p.ppms_specialtycode == entity.CodedSpecialty
                                && p.ppms_VAProviderRelationshipId != null && p.ppms_VAProviderRelationshipId.Id == rel.Id
                                && (p.ppms_caresite != null && csRef.Contains(p.ppms_caresite))
                            );
                        }
                    }
                }
                else
                {
                    // Match by correlation id
                    matches = provider.ppms_account_ppms_providerservice.Where(
                        p => p.StateCode == ppms_providerserviceState.Active
                        && p.Id == new Guid(entity.CorrelationId)
                    );
                }

                // Add matches to target list
                list.AddRange(matches);
            }

            return EntityDelete((IEnumerable<ppms_providerservice>)list);
        }
    }
}